<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Mail\EmailVerificationOtp;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;

class EmailVerificationController extends Controller
{
    /**
     * Send OTP for email verification
     */
    public function sendOtp()
    {
        $user = Auth::user();

        if ($user->hasVerifiedEmail()) {
            return redirect()->route('dashboard')->with('success', 'Your email is already verified.');
        }

        // Generate 6-digit OTP
        $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);

        // Update user with OTP and expiration time (15 minutes)
        $user->update([
            'email_verification_code' => $otp,
            'email_verification_code_expires_at' => Carbon::now()->addMinutes(15)
        ]);

        try {
            // Send OTP email
            Mail::to($user->email)->send(new EmailVerificationOtp($user, $otp));
            \Log::info("Email verification OTP sent successfully for user: {$user->name} (ID: {$user->id}) - Email: {$user->email}");

            return back()->with('status', 'A verification code has been sent to your email address.');
        } catch (\Exception $e) {
            \Log::error("Failed to send email verification OTP for user: {$user->name} (ID: {$user->id}) - Email: {$user->email}. Error: " . $e->getMessage());

            return back()->with('error', 'Failed to send verification code. Please try again.');
        }
    }

    /**
     * Verify OTP
     */
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'verification_code' => 'required|string|size:6'
        ]);

        $user = Auth::user();

        if ($user->hasVerifiedEmail()) {
            return redirect()->route('dashboard')->with('success', 'Your email is already verified.');
        }

        // Check if OTP matches and hasn't expired
        if ($user->email_verification_code === $request->verification_code) {
            if (Carbon::now()->lessThanOrEqualTo($user->email_verification_code_expires_at)) {
                // Mark email as verified
                $user->update([
                    'email_verified_at' => Carbon::now(),
                    'email_verification_code' => null,
                    'email_verification_code_expires_at' => null
                ]);

                \Log::info("Email verification completed successfully for user: {$user->name} (ID: {$user->id}) - Email: {$user->email}");

                return redirect()->route('dashboard')->with('success', 'Your email has been verified successfully!');
            } else {
                return back()->with('error', 'The verification code has expired. Please request a new one.');
            }
        } else {
            return back()->with('error', 'The verification code is incorrect. Please try again.');
        }
    }
}
